const threesixty = (container, images, options) => {
    if (!container) {
        throw new Error('A container argument is required')
    }

    if (!images) {
        throw new Error('An images argument is required')
    }

    const defaults = {
        interactive: true,
        currentFrame: 0
    }

    const o = Object.assign({}, defaults, options)
    const totalFrames = images.length

    let mouseX = 0
    let oldMouseX = 0
    let dragOrigin = false;
    let dragTolerance = 10;
    let swipeTolerance = 10;
    let loopTimeoutId = null;
    let looping = false;
    let speed = 70;

    //------------------------------------------------------------------------------
    //
    //  Initialisation
    //
    //------------------------------------------------------------------------------

    const init = () => {
        preloadimages(images, start)
    }

    const preloadimages = (sourceImages, cb) => {
        const total = sourceImages.length
        let loaded = 0

        const onload = () => {
            if (++loaded >= total) cb(finalImages)
        }

        const finalImages = sourceImages.map((item) => {
            const image = new Image()
            image.src = item
            image.onload = onload
            image.onerror = onload
            image.onabort = onload
            image.draggable = false
            return image
        })
    }

    const start = (loadedImages) => {
        images = loadedImages

        emptyDomNode(container)
        container.appendChild(images[o.currentFrame])

        initListeners();
        play();
    }

    //------------------------------------------------------------------------------
    //
    //  Events
    //
    //------------------------------------------------------------------------------

    const initListeners = () => {
        //drag
        container.addEventListener('mousedown', function (e) {
            dragOrigin = e.pageX;
        });

        document.addEventListener('mouseup', function () {
            dragOrigin = false;
        });

        document.addEventListener('mousemove', function (e) {
            if (dragOrigin && Math.abs(dragOrigin - e.pageX) > dragTolerance) {
                stop();
                dragOrigin > e.pageX ? previous() : next();
                dragOrigin = e.pageX;
            }
        });

        //swipe
        container.addEventListener('touchstart', function (e) {
            dragOrigin = e.touches[0].clientX;
        });

        container.addEventListener('touchend', function () {
            dragOrigin = false;
        });

        document.addEventListener('touchmove', function (e) {
            if (dragOrigin && Math.abs(dragOrigin - e.touches[0].clientX) > swipeTolerance) {
                stop();
                dragOrigin > e.touches[0].clientX ?  previous() : next();
                dragOrigin = e.touches[0].clientX;
            }
        });
    }



    //------------------------------------------------------------------------------
    //
    //  Sequence management
    //
    //------------------------------------------------------------------------------

    const replaceImage = () => {
        container.replaceChild(images[o.currentFrame], container.childNodes[0])
    }

    const previous = () => {
        o.currentFrame--
        if (o.currentFrame < 0) o.currentFrame = totalFrames - 1
        replaceImage()
    }

    const next = () => {
        o.currentFrame++
        if (o.currentFrame === totalFrames) o.currentFrame = 0
        replaceImage()
    }

    const isInteractive = () => o.interactive
    const getCurrentFrame = () => o.currentFrame

    const play = () => {
        if (looping) {
            return;
        }

        loop(false);
        looping = true;
    }

    const stop = () => {
        if (!looping) {
            return;
        }

        window.clearTimeout(loopTimeoutId);
        looping = false;
    }

    const loop = (reversed) => {
        reversed ? previous() : next();

        loopTimeoutId = window.setTimeout(function () {
            loop(reversed);
        }, speed);
    }

    //------------------------------------------------------------------------------
    //
    //  API
    //
    //------------------------------------------------------------------------------

    return {
        init,
        previous,
        next,
        isInteractive,
        getCurrentFrame,
        play,
        stop,
        loop
    }
}

//------------------------------------------------------------------------------
//
//  Utilities
//
//------------------------------------------------------------------------------

const emptyDomNode = (element) => {
    if (element.hasChildNodes()) {
        while (element.firstChild) {
            element.removeChild(element.firstChild)
        }
    }
}



export default threesixty